function PartieRadiale(Z,n,l,varargin)
%%  PartieRadiale
%
%   Ce programme calcule et trace la partie radiale d'un atome hydrognode, 
%   d'expression gnrale :
%
%   R(n,l)=
% 
%   Entres :   n       = nombre quantique principal
%               l       = moment cintique orbital (l<=n)
%               var     = variable
%   Sorties :   R(n,l)  = Partie Radiale
%
%   On trace R(n,l) et la densit de probabilit radiale r^2|R(n,l)|^2 normalise 
%   Attention : on garantit le calcul jusqu' n=30 --> A voir pour n>30 !
%
% Olivier Pujol : septembre 2013

% clear all
% close all

%% Vrifications et constante

if (n <= 0),error('La valeur de n doit tre strictement positive !');end
if (l < 0),error('La valeur de l doit tre positive ou nulle !');end
if (l >= n),error('Mauvaise valeur de l : il faut l<n !');end
if (Z <= 0),error('Mauvaise valeur de Z : il faut Z>=1 !');end

aB = 0.0529177E-9;  % Rayon de Bohr (m)

%% Dfinition de la maille unidimensionnelle

switch length(varargin)
    case 0
        NbPoints = 201;     % Nombre de points
        k=20;
    case 1
        NbPoints = varargin{1};
        k=20;
    case 2
        NbPoints=varargin{1};
        k=varargin{2};
end

x = linspace(0,k*n/Z,n*NbPoints);     % Variable sans unit r/aB
%Remarques :    1) L'extension de la fonction radiale augmente avec n et
%                   diminue avec Z--> xmax=20*n/Z
%               2) Il n'est pas judicieux de travailler avec x=r/rc avec
%                   rc=naB/Z car alors on ne voit pas graphiquement l'effet de Z.

%% Calcul du polynme associ de Laguerre L(n+l,2l+1)

PaLag=PolyAssoLaguerre(n,l,2*Z*x/n);          % Polynme associ de Laguerre

%% Calcul de la partie radiale R(n,l)

Facteur =(2*Z/(n*aB))^(1.5)*(gamma(n-l)/(2*n*(gamma(n+l+1))^3))^(0.5);

Rnl=-Facteur*(2*Z*x/n).^l.*exp(-Z*x/n).*PaLag;    % Partie radiale

%% Calcul de la densit radiale de probabilit

DrProba=aB.^2.*(abs(Rnl)).^2.*x.^2;           % |Rnl|^2 r^2 dr = |Rnl|^2 aB^2 x^2 dr 

%% Reprsentations graphiques

% 1) Partie radiale R(n,l)

figure;

subplot(2,1,1)
plot(x,Rnl*aB^(1.5),'r')    % On trace Rnl*aB^1.5 (et non Rnl) pour viter les puissances de 10
set(gca,'xminortick','on','yminortick','on');
grid on
box on
axis tight
xlabel('r/a_B','Fontsize',12);
title(['a_B^{3/2}  R(n,l) avec n = ',num2str(n),' et l = ',num2str(l),''],'Fontsize',12);

% 2) Densit radiale de probabilit r^2|R(n,l)|^2

subplot(2,1,2)
plot(x,DrProba*aB,'r')  % On multiplie par aB pour viter les puissances de 10
set(gca,'xminortick','on','yminortick','on');
grid on
box on
axis tight
xlabel('r/a_B','Fontsize',12);
title(['a_B  r^2  |R(n,l)|^2 avec n = ',num2str(n),' et l = ',num2str(l),''],'Fontsize',12);
